<?php
/**
 * Geolocation System - Unified IP Detection and Location Data
 * Supports both PHP include and AJAX JSON endpoint
 * Integrated cache system for optimal performance
 * 
 * @author Enhanced System
 * @version 3.0
 */

// ========================================
// INITIALIZATION & CONFIGURATION
// ========================================

// Check request type
$is_ajax_request = isset($_GET['ajax']) && $_GET['ajax'] == '1';
$return_json = isset($_GET['format']) && $_GET['format'] == 'json';

// Global session cache key
$global_session_key = 'unified_geolocation_cache';

// Initialize fallback values (ALWAYS SET)
$lenzz_ipaddress = $_SERVER['REMOTE_ADDR'] ?? 'Unknown';
$lenzz_country = 'Unknown';
$lenzz_countryCode = 'XX';
$lenzz_regionName = 'Unknown';
$lenzz_city = 'Unknown';
$lenzz_timezone = 'Unknown';
$lenzz_isp = 'Unknown';
$lenzz_as = 'Unknown';
$lenzz_flag = '🌐';
$lenzz_callcode = '';

// ========================================
// HELPER FUNCTIONS
// ========================================

/**
 * Get client IP address with proxy support
 * @return string Client IP address
 */
function getClientIP()
{
    $ipaddress = '';
    
    if (isset($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $ipaddress = $_SERVER['HTTP_X_FORWARDED_FOR'];
    } elseif (isset($_SERVER['HTTP_CLIENT_IP'])) {
        $ipaddress = $_SERVER['HTTP_CLIENT_IP'];
    } elseif (isset($_SERVER['REMOTE_ADDR'])) {
        $ipaddress = $_SERVER['REMOTE_ADDR'];
    } else {
        $ipaddress = 'UNKNOWN';
    }
    
    // Handle multiple IPs (take first one)
    $ipaddress = explode(',', $ipaddress)[0];
    return trim($ipaddress);
}

/**
 * Fetch geolocation data from API service
 * @param array $service API service configuration
 * @param string $ip IP address to lookup
 * @return array|null Geolocation data or null on failure
 */
function fetchGeolocationData($service, $ip)
{
    $curl = @curl_init($service['url']);
    if ($curl === false) {
        return null;
    }
    
    curl_setopt($curl, CURLOPT_URL, $service['url']);
    curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($curl, CURLOPT_TIMEOUT, $service['timeout']);
    curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, 3);
    curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($curl, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    
    $response = @curl_exec($curl);
    $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($curl);
    curl_close($curl);
    
    if (!$response || $curl_error || $http_code != 200) {
        return null;
    }
    
    $data = json_decode($response, true);
    if (!$data) {
        return null;
    }
    
    return normalizeApiResponse($data, $service['name']);
}

/**
 * Normalize API response to standard format
 * @param array $data Raw API response
 * @param string $source API source name
 * @return array|null Normalized data or null
 */
function normalizeApiResponse($data, $source)
{
    switch ($source) {
        case 'ip-api.com':
            if (isset($data['country'])) {
                return $data;
            }
            break;
            
        case 'ipapi.co':
            if (isset($data['country_name'])) {
                return [
                    'country' => $data['country_name'] ?? 'Unknown',
                    'countryCode' => $data['country_code'] ?? 'XX',
                    'regionName' => $data['region'] ?? 'Unknown',
                    'city' => $data['city'] ?? 'Unknown',
                    'timezone' => $data['timezone'] ?? 'Unknown',
                    'isp' => $data['org'] ?? 'Unknown',
                    'as' => $data['asn'] ?? 'Unknown'
                ];
            }
            break;
            
        case 'ipinfo.io':
            if (isset($data['region'])) {
                return [
                    'country' => $data['country'] ?? 'Unknown',
                    'countryCode' => $data['country'] ?? 'XX',
                    'regionName' => $data['region'] ?? 'Unknown',
                    'city' => $data['city'] ?? 'Unknown',
                    'timezone' => $data['timezone'] ?? 'Unknown',
                    'isp' => $data['org'] ?? 'Unknown',
                    'as' => $data['org'] ?? 'Unknown'
                ];
            }
            break;
            
        case 'freegeoip.app':
            if (isset($data['country_name'])) {
                return [
                    'country' => $data['country_name'] ?? 'Unknown',
                    'countryCode' => $data['country_code'] ?? 'XX',
                    'regionName' => $data['region_name'] ?? 'Unknown',
                    'city' => $data['city'] ?? 'Unknown',
                    'timezone' => $data['time_zone'] ?? 'Unknown',
                    'isp' => 'Unknown',
                    'as' => 'Unknown'
                ];
            }
            break;
    }
    
    return null;
}

// ========================================
// MAIN PROCESSING
// ========================================

// Get primary IP (always fresh)
$primary_ip = getClientIP();
$lenzz_ipaddress = $primary_ip;

// Try to start session safely
try {
    if (session_status() == PHP_SESSION_NONE) {
        session_start();
    }
    
    // Detect if this is a page refresh
    $is_page_refresh = !$is_ajax_request && (
        !isset($_SESSION['page_load_time']) || 
        (isset($_SERVER['HTTP_CACHE_CONTROL']) && $_SERVER['HTTP_CACHE_CONTROL'] === 'no-cache') ||
        (isset($_SERVER['HTTP_PRAGMA']) && $_SERVER['HTTP_PRAGMA'] === 'no-cache')
    );
    
    if ($is_page_refresh && !$is_ajax_request) {
        $_SESSION['page_load_time'] = time();
    }
} catch (Exception $e) {
    $is_page_refresh = true;
}

// ========================================
// CACHE MANAGEMENT
// ========================================

$use_cached = false;

if (isset($_SESSION[$global_session_key])) {
    $cached_data = $_SESSION[$global_session_key];
    $cache_age = time() - $cached_data['timestamp'];
    
    if ($is_ajax_request) {
        // AJAX calls prefer cache for performance (15 minutes)
        if ($cache_age < 900) {
            $use_cached = true;
        }
    } else {
        // Page refresh allows stricter caching rules
        $ip_changed = ($cached_data['ip'] !== $primary_ip);
        
        if ($ip_changed && $cache_age > 60) {
            // IP changed and cache is old - fetch fresh data
            $use_cached = false;
        } elseif ($cache_age < 300) {
            // Same IP or fresh cache - use cached data (5 minutes)
            $use_cached = true;
        }
    }
    
    if ($use_cached) {
        // Use cached geolocation data, IP is always fresh
        $lenzz_country = $cached_data['country'];
        $lenzz_countryCode = $cached_data['countryCode'];
        $lenzz_regionName = $cached_data['regionName'];
        $lenzz_city = $cached_data['city'];
        $lenzz_timezone = $cached_data['timezone'];
        $lenzz_isp = $cached_data['isp'];
        $lenzz_as = $cached_data['as'];
        $lenzz_flag = $cached_data['flag'];
        $lenzz_callcode = $cached_data['callcode'];
    }
}

// ========================================
// API DATA FETCHING
// ========================================

if (!$use_cached) {
    // Store fallback data from previous cache if IP changed
    $fallback_country = $lenzz_country;
    $fallback_countryCode = $lenzz_countryCode;
    $fallback_flag = $lenzz_flag;
    $fallback_callcode = $lenzz_callcode;
    
    // Define API services with fallback order
    $api_services = [
        [
            'name' => 'ip-api.com',
            'url' => "http://ip-api.com/json/$primary_ip",
            'timeout' => 8
        ],
        [
            'name' => 'ipapi.co',
            'url' => "https://ipapi.co/$primary_ip/json/",
            'timeout' => 6
        ],
        [
            'name' => 'ipinfo.io',
            'url' => "https://ipinfo.io/$primary_ip/json",
            'timeout' => 6
        ],
        [
            'name' => 'freegeoip.app',
            'url' => "https://freegeoip.app/json/$primary_ip",
            'timeout' => 5
        ]
    ];
    
    $data = null;
    $successful_api = '';
    
    // Try each API service until we get valid data
    foreach ($api_services as $service) {
        $data = fetchGeolocationData($service, $primary_ip);
        if ($data && isset($data['country'])) {
            $successful_api = $service['name'];
            break;
        }
    }
    
    // Update variables with fresh data or use fallbacks
    if ($data && isset($data['country']) && $successful_api) {
        $lenzz_country = $data['country'] ?? 'Unknown';
        $lenzz_countryCode = $data['countryCode'] ?? 'XX';
        $lenzz_regionName = $data['regionName'] ?? 'Unknown';
        $lenzz_city = $data['city'] ?? 'Unknown';
        $lenzz_timezone = $data['timezone'] ?? 'Unknown';
        $lenzz_isp = $data['isp'] ?? 'Unknown';
        $lenzz_as = $data['as'] ?? 'Unknown';
    } else {
        // Use fallback data from previous cache for consistency
        if ($fallback_country !== 'Unknown' && $fallback_countryCode !== 'XX') {
            $lenzz_country = $fallback_country;
            $lenzz_countryCode = $fallback_countryCode;
            $lenzz_flag = $fallback_flag;
            $lenzz_callcode = $fallback_callcode;
        }
    }
}

// ========================================
// FLAGS & CALLING CODES MAPPING
// ========================================

$flags = [
    "AC" => "🇦🇨", "AD" => "🇦🇩", "AE" => "🇦🇪", "AF" => "🇦🇫",
    "AG" => "🇦🇬", "AI" => "🇦🇮", "AL" => "🇦🇱", "AM" => "🇦🇲",
    "AO" => "🇦🇴", "AQ" => "🇦🇶", "AR" => "🇦🇷", "AS" => "🇦🇸",
    "AT" => "🇦🇹", "AU" => "🇦🇺", "AW" => "🇦🇼", "AX" => "🇦🇽",
    "AZ" => "🇦🇿", "BA" => "🇧🇦", "BB" => "🇧🇧", "BD" => "🇧🇩",
    "BE" => "🇧🇪", "BF" => "🇧🇫", "BG" => "🇧🇬", "BH" => "🇧🇭",
    "BI" => "🇧🇮", "BJ" => "🇧🇯", "BL" => "🇧🇱", "BM" => "🇧🇲",
    "BN" => "🇧🇳", "BO" => "🇧🇴", "BQ" => "🇧🇶", "BR" => "🇧🇷",
    "BS" => "🇧🇸", "BT" => "🇧🇹", "BV" => "🇧🇻", "BW" => "🇧🇼",
    "BY" => "🇧🇾", "BZ" => "🇧🇿", "CA" => "🇨🇦", "CC" => "🇨🇨",
    "CD" => "🇨🇩", "CF" => "🇨🇫", "CG" => "🇨🇬", "CH" => "🇨🇭",
    "CI" => "🇨🇮", "CK" => "🇨🇰", "CL" => "🇨🇱", "CM" => "🇨🇲",
    "CN" => "🇨🇳", "CO" => "🇨🇴", "CP" => "🇨🇵", "CR" => "🇨🇷",
    "CU" => "🇨🇺", "CV" => "🇨🇻", "CW" => "🇨🇼", "CX" => "🇨🇽",
    "CY" => "🇨🇾", "CZ" => "🇨🇿", "DE" => "🇩🇪", "DG" => "🇩🇬",
    "DJ" => "🇩🇯", "DK" => "🇩🇰", "DM" => "🇩🇲", "DO" => "🇩🇴",
    "DZ" => "🇩🇿", "EA" => "🇪🇦", "EC" => "🇪🇨", "EE" => "🇪🇪",
    "EG" => "🇪🇬", "EH" => "🇪🇭", "ER" => "🇪🇷", "ES" => "🇪🇸",
    "ET" => "🇪🇹", "EU" => "🇪🇺", "FI" => "🇫🇮", "FJ" => "🇫🇯",
    "FK" => "🇫🇰", "FM" => "🇫🇲", "FO" => "🇫🇴", "FR" => "🇫🇷",
    "GA" => "🇬🇦", "GB" => "🇬🇧", "GD" => "🇬🇩", "GE" => "🇬🇪",
    "GF" => "🇬🇫", "GG" => "🇬🇬", "GH" => "🇬🇭", "GI" => "🇬🇮",
    "GL" => "🇬🇱", "GM" => "🇬🇲", "GN" => "🇬🇳", "GP" => "🇬🇵",
    "GQ" => "🇬🇶", "GR" => "🇬🇷", "GS" => "🇬🇸", "GT" => "🇬🇹",
    "GU" => "🇬🇺", "GW" => "🇬🇼", "GY" => "🇬🇾", "HK" => "🇭🇰",
    "HM" => "🇭🇲", "HN" => "🇭🇳", "HR" => "🇭🇷", "HT" => "🇭🇹",
    "HU" => "🇭🇺", "IC" => "🇮🇨", "ID" => "🇮🇩", "IE" => "🇮🇪",
    "IL" => "🇮🇱", "IM" => "🇮🇲", "IN" => "🇮🇳", "IO" => "🇮🇴",
    "IQ" => "🇮🇶", "IR" => "🇮🇷", "IS" => "🇮🇸", "IT" => "🇮🇹",
    "JE" => "🇯🇪", "JM" => "🇯🇲", "JO" => "🇯🇴", "JP" => "🇯🇵",
    "KE" => "🇰🇪", "KG" => "🇰🇬", "KH" => "🇰🇭", "KI" => "🇰🇮",
    "KM" => "🇰🇲", "KN" => "🇰🇳", "KP" => "🇰🇵", "KR" => "🇰🇷",
    "KW" => "🇰🇼", "KY" => "🇰🇾", "KZ" => "🇰🇿", "LA" => "🇱🇦",
    "LB" => "🇱🇧", "LC" => "🇱🇨", "LI" => "🇱🇮", "LK" => "🇱🇰",
    "LR" => "🇱🇷", "LS" => "🇱🇸", "LT" => "🇱🇹", "LU" => "🇱🇺",
    "LV" => "🇱🇻", "LY" => "🇱🇾", "MA" => "🇲🇦", "MC" => "🇲🇨",
    "MD" => "🇲🇩", "ME" => "🇲🇪", "MF" => "🇲🇫", "MG" => "🇲🇬",
    "MH" => "🇲🇭", "MK" => "🇲🇰", "ML" => "🇲🇱", "MM" => "🇲🇲",
    "MN" => "🇲🇳", "MO" => "🇲🇴", "MP" => "🇲🇵", "MQ" => "🇲🇶",
    "MR" => "🇲🇷", "MS" => "🇲🇸", "MT" => "🇲🇹", "MU" => "🇲🇺",
    "MV" => "🇲🇻", "MW" => "🇲🇼", "MX" => "🇲🇽", "MY" => "🇲🇾",
    "MZ" => "🇲🇿", "NA" => "🇳🇦", "NC" => "🇳🇨", "NE" => "🇳🇪",
    "NF" => "🇳🇫", "NG" => "🇳🇬", "NI" => "🇳🇮", "NL" => "🇳🇱",
    "NO" => "🇳🇴", "NP" => "🇳🇵", "NR" => "🇳🇷", "NU" => "🇳🇺",
    "NZ" => "🇳🇿", "OM" => "🇴🇲", "PA" => "🇵🇦", "PE" => "🇵🇪",
    "PF" => "🇵🇫", "PG" => "🇵🇬", "PH" => "🇵🇭", "PK" => "🇵🇰",
    "PL" => "🇵🇱", "PM" => "🇵🇲", "PN" => "🇵🇳", "PR" => "🇵🇷",
    "PS" => "🇵🇸", "PT" => "🇵🇹", "PW" => "🇵🇼", "PY" => "🇵🇾",
    "QA" => "🇶🇦", "RE" => "🇷🇪", "RO" => "🇷🇴", "RS" => "🇷🇸",
    "RU" => "🇷🇺", "RW" => "🇷🇼", "SA" => "🇸🇦", "SB" => "🇸🇧",
    "SC" => "🇸🇨", "SD" => "🇸🇩", "SE" => "🇸🇪", "SG" => "🇸🇬",
    "SH" => "🇸🇭", "SI" => "🇸🇮", "SJ" => "🇸🇯", "SK" => "🇸🇰",
    "SL" => "🇸🇱", "SM" => "🇸🇲", "SN" => "🇸🇳", "SO" => "🇸🇴",
    "SR" => "🇸🇷", "SS" => "🇸🇸", "ST" => "🇸🇹", "SV" => "🇸🇻",
    "SX" => "🇸🇽", "SY" => "🇸🇾", "SZ" => "🇸🇿", "TA" => "🇹🇦",
    "TC" => "🇹🇨", "TD" => "🇹🇩", "TF" => "🇹🇫", "TG" => "🇹🇬",
    "TH" => "🇹🇭", "TJ" => "🇹🇯", "TK" => "🇹🇰", "TL" => "🇹🇱",
    "TM" => "🇹🇲", "TN" => "🇹🇳", "TO" => "🇹🇴", "TR" => "🇹🇷",
    "TT" => "🇹🇹", "TV" => "🇹🇻", "TW" => "🇹🇼", "TZ" => "🇹🇿",
    "UA" => "🇺🇦", "UG" => "🇺🇬", "UM" => "🇺🇲", "UN" => "🇺🇳",
    "US" => "🇺🇸", "UY" => "🇺🇾", "UZ" => "🇺🇿", "VA" => "🇻🇦",
    "VC" => "🇻🇨", "VE" => "🇻🇪", "VG" => "🇻🇬", "VI" => "🇻🇮",
    "VN" => "🇻🇳", "VU" => "🇻🇺", "WF" => "🇼🇫", "WS" => "🇼🇸",
    "XK" => "🇽🇰", "YE" => "🇾🇪", "YT" => "🇾🇹", "ZA" => "🇿🇦",
    "ZM" => "🇿🇲", "ZW" => "🇿🇼"
];

$calling_codes = [
    "AD" => "376", "AE" => "971", "AF" => "93", "AG" => "1-268",
    "AI" => "1-264", "AL" => "355", "AM" => "374", "AO" => "244",
    "AQ" => "", "AR" => "54", "AS" => "1-684", "AT" => "43",
    "AU" => "61", "AW" => "297", "AX" => "358-18", "AZ" => "994",
    "BA" => "387", "BB" => "1-246", "BD" => "880", "BE" => "32",
    "BF" => "226", "BG" => "359", "BH" => "973", "BI" => "257",
    "BJ" => "229", "BL" => "590", "BM" => "1-441", "BN" => "673",
    "BO" => "591", "BQ" => "599", "BR" => "55", "BS" => "1-242",
    "BT" => "975", "BV" => "", "BW" => "267", "BY" => "375",
    "BZ" => "501", "CA" => "1", "CC" => "61", "CD" => "243",
    "CF" => "236", "CG" => "242", "CH" => "41", "CI" => "225",
    "CK" => "682", "CL" => "56", "CM" => "237", "CN" => "86",
    "CO" => "57", "CR" => "506", "CU" => "53", "CV" => "238",
    "CW" => "599", "CX" => "61", "CY" => "357", "CZ" => "420",
    "DE" => "49", "DJ" => "253", "DK" => "45", "DM" => "1-767",
    "DO" => "1-809and1-829", "DZ" => "213", "EC" => "593", "EE" => "372",
    "EG" => "20", "EH" => "212", "ER" => "291", "ES" => "34",
    "ET" => "251", "FI" => "358", "FJ" => "679", "FK" => "500",
    "FM" => "691", "FO" => "298", "FR" => "33", "GA" => "241",
    "GB" => "44", "GD" => "1-473", "GE" => "995", "GF" => "594",
    "GG" => "44-1481", "GH" => "233", "GI" => "350", "GL" => "299",
    "GM" => "220", "GN" => "224", "GP" => "590", "GQ" => "240",
    "GR" => "30", "GS" => "", "GT" => "502", "GU" => "1-671",
    "GW" => "245", "GY" => "592", "HK" => "852", "HM" => "",
    "HN" => "504", "HR" => "385", "HT" => "509", "HU" => "36",
    "ID" => "62", "IE" => "353", "IL" => "972", "IM" => "44-1624",
    "IN" => "91", "IO" => "246", "IQ" => "964", "IR" => "98",
    "IS" => "354", "IT" => "39", "JE" => "44-1534", "JM" => "1-876",
    "JO" => "962", "JP" => "81", "KE" => "254", "KG" => "996",
    "KH" => "855", "KI" => "686", "KM" => "269", "KN" => "1-869",
    "KP" => "850", "KR" => "82", "KW" => "965", "KY" => "1-345",
    "KZ" => "7", "LA" => "856", "LB" => "961", "LC" => "1-758",
    "LI" => "423", "LK" => "94", "LR" => "231", "LS" => "266",
    "LT" => "370", "LU" => "352", "LV" => "371", "LY" => "218",
    "MA" => "212", "MC" => "377", "MD" => "373", "ME" => "382",
    "MF" => "590", "MG" => "261", "MH" => "692", "MK" => "389",
    "ML" => "223", "MM" => "95", "MN" => "976", "MO" => "853",
    "MP" => "1-670", "MQ" => "596", "MR" => "222", "MS" => "1-664",
    "MT" => "356", "MU" => "230", "MV" => "960", "MW" => "265",
    "MX" => "52", "MY" => "60", "MZ" => "258", "NA" => "264",
    "NC" => "687", "NE" => "227", "NF" => "672", "NG" => "234",
    "NI" => "505", "NL" => "31", "NO" => "47", "NP" => "977",
    "NR" => "674", "NU" => "683", "NZ" => "64", "OM" => "968",
    "PA" => "507", "PE" => "51", "PF" => "689", "PG" => "675",
    "PH" => "63", "PK" => "92", "PL" => "48", "PM" => "508",
    "PN" => "870", "PR" => "1-787and1-939", "PS" => "970", "PT" => "351",
    "PW" => "680", "PY" => "595", "QA" => "974", "RE" => "262",
    "RO" => "40", "RS" => "381", "RU" => "7", "RW" => "250",
    "SA" => "966", "SB" => "677", "SC" => "248", "SD" => "249",
    "SE" => "46", "SG" => "65", "SH" => "290", "SI" => "386",
    "SJ" => "47", "SK" => "421", "SL" => "232", "SM" => "378",
    "SN" => "221", "SO" => "252", "SR" => "597", "SS" => "211",
    "ST" => "239", "SV" => "503", "SX" => "599", "SY" => "963",
    "SZ" => "268", "TC" => "1-649", "TD" => "235", "TF" => "",
    "TG" => "228", "TH" => "66", "TJ" => "992", "TK" => "690",
    "TL" => "670", "TM" => "993", "TN" => "216", "TO" => "676",
    "TR" => "90", "TT" => "1-868", "TV" => "688", "TW" => "886",
    "TZ" => "255", "UA" => "380", "UG" => "256", "UM" => "1",
    "US" => "1", "UY" => "598", "UZ" => "998", "VA" => "379",
    "VC" => "1-784", "VE" => "58", "VG" => "1-284", "VI" => "1-340",
    "VN" => "84", "VU" => "678", "WF" => "681", "WS" => "685",
    "XK" => "383", "YE" => "967", "YT" => "262", "ZA" => "27",
    "ZM" => "260", "ZW" => "263"
];

// Set flag and calling code
$lenzz_flag = isset($flags[$lenzz_countryCode]) ? $flags[$lenzz_countryCode] : '🌐';
$lenzz_callcode = isset($calling_codes[$lenzz_countryCode]) ? $calling_codes[$lenzz_countryCode] : '';

// ========================================
// CACHE UPDATE
// ========================================

// Update cache with fresh data
if (!$use_cached && isset($_SESSION)) {
    try {
        $_SESSION[$global_session_key] = [
            'ip' => $lenzz_ipaddress,
            'country' => $lenzz_country,
            'countryCode' => $lenzz_countryCode,
            'regionName' => $lenzz_regionName,
            'city' => $lenzz_city,
            'timezone' => $lenzz_timezone,
            'isp' => $lenzz_isp,
            'as' => $lenzz_as,
            'flag' => $lenzz_flag,
            'callcode' => $lenzz_callcode,
            'timestamp' => time()
        ];
    } catch (Exception $e) {
        // Silent fail for production
    }
}

// ========================================
// OUTPUT HANDLING
// ========================================

// If this is an AJAX request, return JSON response
if ($is_ajax_request && $return_json) {
    header('Content-Type: application/json; charset=utf-8');
    header('Cache-Control: no-cache, must-revalidate');
    header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');
    
    $response = [
        'status' => 'success',
        'data' => [
            'ip' => $lenzz_ipaddress,
            'country' => $lenzz_country,
            'countryCode' => $lenzz_countryCode,
            'regionName' => $lenzz_regionName,
            'city' => $lenzz_city,
            'timezone' => $lenzz_timezone,
            'isp' => $lenzz_isp,
            'as' => $lenzz_as,
            'flag' => $lenzz_flag,
            'callcode' => $lenzz_callcode
        ],
        'cache_info' => [
            'used_cache' => $use_cached ?? false,
            'timestamp' => time(),
            'source' => $use_cached ? 'cache' : 'fresh_api'
        ]
    ];
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    exit();
}

// For PHP includes, variables are already set and ready to use
?>